/*
 * fifo.c - syslogd implementation for windows, just a queue
 *          for the replacement of glib's queue.
 *
 * Created by Alexander Yaworsky
 *
 * THIS SOFTWARE IS NOT COPYRIGHTED
 *
 * This source code is offered for use in the public domain. You may
 * use, modify or distribute it freely.
 *
 * This code is distributed in the hope that it will be useful but
 * WITHOUT ANY WARRANTY. ALL WARRANTIES, EXPRESS OR IMPLIED ARE HEREBY
 * DISCLAIMED. This includes but is not limited to warranties of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <windows.h>
#include <glib.h>
#include <syslog.h>
#include <syslogd.h>

struct fifo_item
{
    struct fifo_item *next;  /* queue is a single-linked list */
    void *payload;
};

struct fifo
{
    struct fifo_item *first;  /* first pushed item */
    struct fifo_item *last;   /* last pushed item */
    int item_count;
};

/******************************************************************************
 * fifo_create
 *
 * Allocate and initialize fifo structure. Add an empty item to the fifo.
 */
struct fifo* fifo_create()
{
    struct fifo *ret = g_malloc( sizeof(struct fifo) );
    ret->first = NULL;
    ret->last = NULL;
    ret->item_count = 0;
    return ret;
}

/******************************************************************************
 * fifo_destroy
 *
 * Delete all items and free fifo structure.
 */
void fifo_destroy( struct fifo* queue )
{
    struct fifo_item *item;

    while( (item = queue->first) != NULL )
    {
        queue->first = item->next;
        g_free( item );
    }
    g_free( queue );
}

/******************************************************************************
 * fifo_push
 *
 * Add item to queue.
 * Return TRUE if added item is the first in the queue.
 */
gboolean fifo_push( struct fifo* queue, void* data )
{
    struct fifo_item *item = g_malloc( sizeof(struct fifo_item) );
    item->next = NULL;
    item->payload = data;
    if( !queue->last )
        queue->first = item;
    else
        queue->last->next = item;
    queue->last = item;
    return 1 == ++queue->item_count;
}

/******************************************************************************
 * fifo_pop
 *
 * Extract item from queue.
 */
void* fifo_pop( struct fifo* queue )
{
    struct fifo_item *item;
    void *data;

    item = queue->first;
    if( !item )
        return NULL;

    queue->first = item->next;
    if( !queue->first )
        queue->last = NULL;

    data = item->payload;
    g_free( item );
    queue->item_count--;
    return data;
}
